import React from "react";
import classnames from "classnames";
import moment from "moment";
import { Table, Button, Pagination, Drawer, Form, Field, Input, NumberPicker, Dialog, DatePicker2, Select, Grid } from "@alifd/next";
import "./index.scss";
import { createOrder, deleteOrder, queryOrder, queryOrderList, updateOrder } from "../../service/order";

const { Row, Col } = Grid;

type DataSource<T> = {
    items: T[];
    totalCount: number;
};

const Column = Table.Column;

type ComponentProps = {};

type ComponentState = {
    editVisible: boolean;
    viewVisible: boolean;
    currentPage: number;
    pageSize: number;
    dataSource: DataSource<any>;
    sortName?: string;
    sortType?: "asc" | "desc";
};

const statusDataSource = [
    {
        lable: "待发货",
        value: "待发货",
    },
    {
        lable: "已发货",
        value: "已发货",
    },
    {
        lable: "交易成功",
        value: "交易成功",
    },
];

class OrderItems extends React.Component<any, any> {
    render() {
        const { value } = this.props;

        return (
            <div>
                <Table dataSource={value}>
                    <Column title="商品名称" dataIndex="commodityName" />
                    <Column title="商品数量" dataIndex="commodityQuantity" />
                    <Column title="商品价格" dataIndex="commodityPrice" />
                </Table>
            </div>
        );
    }
}

export default class View extends React.Component<ComponentProps, ComponentState> {
    private readonly field: Field;
    private readonly queryField: Field;

    constructor(props: ComponentProps) {
        super(props);

        this.state = {
            currentPage: 1,
            pageSize: 10,
            editVisible: false,
            viewVisible: false,
            dataSource: {
                items: [],
                totalCount: 0,
            },
        };

        this.field = new Field(this);
        this.queryField = new Field(this);
    }

    componentDidMount() {
        this.fetchData({ currentPage: 1 });
    }

    fetchData(params?: { pageSize?: number; currentPage?: number }) {
        const { pageSize, currentPage, sortName, sortType } = this.state;
        const pageNumber = params?.currentPage || currentPage;
        const _pageSize = params?.pageSize || pageSize;

        queryOrderList({
            pageNumber,
            pageSize: _pageSize,
            sortName,
            sortType,
            condition: this.queryField.getValues(),
        }).then((result) => {
            this.setState({
                dataSource: result,
                currentPage: pageNumber,
                pageSize: _pageSize,
            });
        });
    }

    onSort(dataIndex, order) {
        this.setState(
            {
                sortName: dataIndex,
                sortType: order,
            },
            () => {
                this.fetchData();
            },
        );
    }

    render() {
        const { dataSource, currentPage, pageSize, editVisible, viewVisible, sortName, sortType } = this.state;
        const values: any = this.field.getValues();

        return (
            <div className="page-wrap">
                <div>
                    <Form field={this.queryField}>
                        <Row gutter={"8"}>
                            <Col span="6">
                                <Form.Item label="订单号" name="id" labelAlign={"left"}>
                                    <Input></Input>
                                </Form.Item>
                            </Col>
                            <Col span="6">
                                <Form.Item label="标题" name="title" labelAlign={"left"}>
                                    <Input></Input>
                                </Form.Item>
                            </Col>
                            <Col span="6">
                                <Form.Item label="状态" name="status" labelAlign={"left"}>
                                    <Select dataSource={statusDataSource} className="field-full" hasClear></Select>
                                </Form.Item>
                            </Col>
                            <Col span="6">
                                <Form.Item label="金额" name="amount">
                                    <NumberPicker className="field-full"></NumberPicker>
                                </Form.Item>
                            </Col>
                        </Row>
                    </Form>
                </div>
                <div className="button-wrap">
                    <Button
                        type="primary"
                        onClick={() => {
                            this.fetchData({ currentPage: 1 });
                        }}
                    >
                        查询
                    </Button>
                    <Button
                        type="primary"
                        onClick={() => {
                            this.queryField.reset();
                            this.fetchData({ currentPage: 1 });
                        }}
                    >
                        清空条件
                    </Button>
                    <Button
                        type="secondary"
                        onClick={() => {
                            this.field.reset();
                            this.setState({
                                editVisible: true,
                            });
                        }}
                    >
                        新增订单
                    </Button>
                </div>
                <div className="table-wrap">
                    <Table dataSource={dataSource.items} onSort={this.onSort.bind(this)} sort={{ [sortName]: sortType }}>
                        <Column title="订单号" dataIndex="id" sortable />
                        <Column title="标题" dataIndex="title" />
                        <Column
                            title="状态"
                            dataIndex="status"
                            cell={(value) => {
                                let className;

                                if (value == "已发货") {
                                    className = "table-cell-status-1";
                                } else if (value == "待发货") {
                                    className = "table-cell-status-2";
                                } else {
                                    className = "table-cell-status-3";
                                }

                                return <div className={classnames("table-cell-status", className)}>{value}</div>;
                            }}
                        />
                        <Column
                            title="金额"
                            dataIndex="amount"
                            sortable
                            cell={(value) => {
                                return `￥ ${value}`;
                            }}
                        />
                        <Column
                            title="创建日期"
                            dataIndex="createDate"
                            sortable
                            cell={(value) => {
                                return value ? moment(value).format("YYYY/MM/DD HH:mm:ss") : "";
                            }}
                        />
                        <Column
                            width={200}
                            title="操作"
                            lock="right"
                            cell={(value: any, index: number, rowData: any) => {
                                return (
                                    <>
                                        <Button
                                            className="table-cell-action"
                                            text={true}
                                            onClick={() => {
                                                queryOrder(rowData.id).then((data) => {
                                                    this.field.setValues(data);

                                                    this.setState({
                                                        editVisible: true,
                                                    });
                                                });
                                            }}
                                        >
                                            编辑
                                        </Button>
                                        <Button
                                            className="table-cell-action"
                                            text={true}
                                            onClick={() => {
                                                queryOrder(rowData.id).then((data) => {
                                                    this.field.setValues(data);

                                                    this.setState({
                                                        viewVisible: true,
                                                    });
                                                });
                                            }}
                                        >
                                            查看
                                        </Button>
                                        <Button
                                            className="table-cell-action"
                                            text={true}
                                            onClick={() => {
                                                Dialog.confirm({
                                                    content: "确认删除吗？",
                                                    onOk: () => {
                                                        deleteOrder(rowData.id).then(() => {
                                                            this.fetchData();
                                                        });
                                                    },
                                                });
                                            }}
                                        >
                                            删除
                                        </Button>
                                    </>
                                );
                            }}
                        />
                    </Table>
                </div>
                <div className="pagination-wrap">
                    <Pagination
                        total={dataSource.totalCount}
                        current={currentPage}
                        pageSize={pageSize}
                        onChange={(currentPage: number) => {
                            this.fetchData({ currentPage });
                        }}
                    />
                </div>
                <Drawer
                    title={values?.id ? "编辑" : "新增"}
                    visible={editVisible}
                    width={720}
                    onClose={() => {
                        this.setState({ editVisible: false });
                    }}
                >
                    <div className="form_wrap">
                        <Form field={this.field} size="medium" labelCol={{ span: 4 }} wrapperCol={{ span: 14 }}>
                            <Form.Item label="标题" name="title" required requiredMessage={"请输入标题"}>
                                <Input></Input>
                            </Form.Item>
                            <Form.Item label="状态" name="status" required requiredMessage={"请输入状态"}>
                                <Select dataSource={statusDataSource} className="field-full" hasClear></Select>
                            </Form.Item>
                            <Form.Item label="金额" name="amount" required requiredMessage={"请输入金额"}>
                                <NumberPicker className="field-full"></NumberPicker>
                            </Form.Item>
                            <Form.Item label="创建日期" name="createDate" required requiredMessage={"请输入创建日期"}>
                                <DatePicker2 showTime className="field-full" />
                            </Form.Item>
                            <Form.Item label=" ">
                                <Button
                                    type="primary"
                                    onClick={() => {
                                        this.field.validate((errors, values: any) => {
                                            if (errors) {
                                                return;
                                            }

                                            const { createDate } = values;

                                            const _values = {
                                                ...values,
                                                createDate: createDate ? createDate.valueOf() : undefined,
                                            };

                                            let flow: Promise<void>;

                                            if (_values.id) {
                                                flow = updateOrder(_values);
                                            } else {
                                                flow = createOrder(_values);
                                            }

                                            flow.then(() => {
                                                this.setState(
                                                    {
                                                        editVisible: false,
                                                    },
                                                    () => {
                                                        this.field.reset();
                                                        this.fetchData({ currentPage: 1 });
                                                    },
                                                );
                                            });
                                        });
                                    }}
                                >
                                    提交
                                </Button>
                            </Form.Item>
                        </Form>
                    </div>
                </Drawer>
                <Drawer
                    title={"查看"}
                    visible={viewVisible}
                    width={720}
                    onClose={() => {
                        this.setState({ viewVisible: false });
                    }}
                >
                    <div className="form_wrap">
                        <Form field={this.field} size="medium" labelCol={{ span: 4 }} wrapperCol={{ span: 14 }}>
                            <Form.Item label="标题" name="title" isPreview>
                                <Input></Input>
                            </Form.Item>
                            <Form.Item label="状态" name="status" isPreview>
                                <Select dataSource={statusDataSource} className="field-full" hasClear></Select>
                            </Form.Item>
                            <Form.Item label="金额" name="amount" isPreview>
                                <NumberPicker className="field-full"></NumberPicker>
                            </Form.Item>
                            <Form.Item label="创建日期" name="createDate" isPreview>
                                <DatePicker2 showTime className="field-full" />
                            </Form.Item>
                            <Form.Item name="orderItems" wrapperCol={{ span: 24 }}>
                                <OrderItems />
                            </Form.Item>
                        </Form>
                    </div>
                </Drawer>
            </div>
        );
    }
}
