import { isPlainObject, pickBy } from "lodash";
import axios, { AxiosRequestConfig } from "axios";

type CommonObject = { [key: string]: any };
type ResponseConfig = {
    formatter?: (data: any) => any;
    silence?: boolean;
};
type RequestConfig = ResponseConfig & AxiosRequestConfig;

const defaultContentType = "application/json";

function defaultFormatter(data: any) {
    if (data && data.code === "OK") {
        return Promise.resolve(data.data);
    } else {
        return Promise.reject(data);
    }
}

const r20 = /%20/g;

function parseData(data: CommonObject) {
    const pairs = [];

    for (const k in data) {
        let value = data[k];

        if (typeof value !== "undefined" && value !== null) {
            if (typeof value === "object") {
                value = JSON.stringify(value);
            }

            pairs.push(`${encodeURIComponent(k)}=${encodeURIComponent(value)}`);
        }
    }

    return pairs.join("&").replace(r20, "+");
}

function paramsSerializer(params: CommonObject) {
    const parts: string[] = [];

    if (isPlainObject(params)) {
        Object.keys(params).forEach((key) => {
            let val = params[key];

            if (val === null || typeof val === "undefined") {
                return;
            }

            if (isPlainObject(val)) {
                val = JSON.stringify(val);
            }

            parts.push(`${key}=${encodeURIComponent(val)}`);
        });
    }

    return parts.join("&");
}

function request(url: string, config: RequestConfig): Promise<any> {
    const { silence, headers: customHeaders, formatter, ...rest } = config;

    rest.params = {
        ...rest.params,
        t: new Date().getTime(),
    };

    rest.paramsSerializer = paramsSerializer;

    const headers = pickBy(
        {
            "X-Requested-With": "XMLHttpRequest",
            "Content-Type": defaultContentType,
            ...customHeaders,
        },
        function (value) {
            return typeof value !== "undefined";
        },
    );

    const data = rest.data || {};
    rest.data = headers["Content-Type"] == defaultContentType ? JSON.stringify(data) : parseData(data);

    return axios
        .create()
        .request({
            ...rest,
            headers,
            url,
        })
        .then((resp) => {
            const data = resp && resp.data;
            if (!data) {
                return Promise.reject({ message: "response is undefined" });
            }

            return formatter ? formatter(data) : defaultFormatter(data);
        })
        .catch((resp) => {
            return Promise.reject(resp);
        });
}

export function get(url: string, params?: any, config?: RequestConfig) {
    return request(url, {
        ...config,
        params,
        method: "get",
    });
}

export function post(url: string, data?: any, config?: RequestConfig) {
    return request(url, {
        ...config,
        data,
        method: "post",
    });
}
